<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class PayRateMaster extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'pay_rate_master';
    protected $primaryKey = 'pay_rate_id';
    public $incrementing = true;
    protected $keyType = 'int';

    protected $fillable = [
        'emp_type_id', 
        'effective_date',
        'pay_parameters'
    ];

    protected $casts = [
        'effective_date' => 'date',
        'pay_parameters' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime'
    ];

    // Pay Parameter Types
    const DEDUCT = 0;
    const ADD = 1;

    const ADD_DEDUCT_TEXTS = [
        self::DEDUCT => 'Deduct',
        self::ADD => 'Add'
    ];

    protected $appends = [
        'total_components',
        'total_percentage',
        'total_amount_per_day',
        'is_active'
    ];

    /**
     * Get the employee type - CHANGED: tenderWork से employeeType
     */
    public function employeeType()
    {
        return $this->belongsTo(EmployeeTypeMaster::class, 'emp_type_id', 'emp_type_id');
    }

    /**
     * Get total number of components
     */
    public function getTotalComponentsAttribute(): int
    {
        return is_array($this->pay_parameters) ? count($this->pay_parameters) : 0;
    }

    /**
     * Calculate total effective percentage
     */
    public function getTotalPercentageAttribute(): float
    {
        if (!is_array($this->pay_parameters)) {
            return 0;
        }

        $total = 0;
        foreach ($this->pay_parameters as $param) {
            $percentage = $param['percentage'] ?? 0;
            $addDeduct = $param['add_deduct'] ?? self::DEDUCT;

            $effectivePercentage = ($addDeduct == self::DEDUCT) ? -$percentage : $percentage;
            $total += $effectivePercentage;
        }

        return $total;
    }

    /**
     * Calculate total amount per day
     */
    public function getTotalAmountPerDayAttribute(): float
    {
        if (!is_array($this->pay_parameters)) {
            return 0;
        }

        $total = 0;
        foreach ($this->pay_parameters as $param) {
            $amountPerDay = $param['amount_per_day'] ?? 0;
            $addDeduct = $param['add_deduct'] ?? self::DEDUCT;

            $effectiveAmount = ($addDeduct == self::DEDUCT) ? -$amountPerDay : $amountPerDay;
            $total += $effectiveAmount;
        }

        return $total;
    }

    /**
     * Check if pay rate is active
     */
    public function getIsActiveAttribute(): bool
    {
        return is_null($this->deleted_at);
    }

    /**
     * Calculate total amount based on number of days
     */
    public function calculateTotalAmount($numberOfDays): float
    {
        if (!is_array($this->pay_parameters)) {
            return 0;
        }

        $total = 0;
        foreach ($this->pay_parameters as $param) {
            $amountPerDay = $param['amount_per_day'] ?? 0;
            if ($amountPerDay > 0) {
                $total += ($amountPerDay * $numberOfDays);
            }
        }

        return $total;
    }

    /**
     * Calculate total percentage amount based on base amount
     */
    public function calculateTotalPercentageAmount($baseAmount): float
    {
        if (!is_array($this->pay_parameters)) {
            return 0;
        }

        $total = 0;
        foreach ($this->pay_parameters as $param) {
            $percentage = $param['percentage'] ?? 0;
            if ($percentage > 0) {
                $total += ($baseAmount * $percentage) / 100;
            }
        }

        return $total;
    }

    /**
     * Validation rules for creating pay rate - CHANGED: tend_work_id से emp_type_id
     */
    public static function createRules()
    {
        return [
            'emp_type_id' => 'required|exists:employee_type_master,emp_type_id', // CHANGED
            'effective_date' => 'required|date|date_format:Y-m-d',
            'pay_parameters' => 'required|array|min:1',
            'pay_parameters.*.payroll_id' => 'required|exists:payroll_master,payroll_id',
            'pay_parameters.*.applicable_criteria_name' => 'required|string|max:100',
            'pay_parameters.*.add_deduct' => 'required|in:0,1',
            'pay_parameters.*.Lg_Id' => 'required|exists:ledger_master,Lg_Id',
            'pay_parameters.*.percentage' => 'nullable|numeric|min:0|max:100|regex:/^\d+(\.\d{1,2})?$/',
            'pay_parameters.*.amount_per_day' => 'nullable|numeric|min:0|max:999999.99|regex:/^\d+(\.\d{1,2})?$/',
            'pay_parameters.*.calculation_formula' => 'nullable|string|max:255'
        ];
    }

    /**
     * Custom validation messages - CHANGED: tend_work_id से emp_type_id
     */
    public static function validationMessages()
    {
        return [
            'emp_type_id.required' => 'Employee Type is required.', // CHANGED
            'emp_type_id.exists' => 'Selected Employee Type does not exist.', // CHANGED
            'effective_date.required' => 'Effective Date is required.',
            'effective_date.date' => 'Effective Date must be a valid date.',
            'effective_date.date_format' => 'Effective Date must be in YYYY-MM-DD format.',
            'pay_parameters.required' => 'At least one pay parameter is required.',
            'pay_parameters.*.payroll_id.required' => 'Payroll ID is required for each parameter.',
            'pay_parameters.*.payroll_id.exists' => 'Selected Payroll does not exist.',
            'pay_parameters.*.applicable_criteria_name.required' => 'Applicable Criteria Name is required.',
            'pay_parameters.*.applicable_criteria_name.string' => 'Applicable Criteria Name must be a string.',
            'pay_parameters.*.applicable_criteria_name.max' => 'Applicable Criteria Name must not exceed 100 characters.',
            'pay_parameters.*.add_deduct.required' => 'Add/Deduct type is required for each parameter.',
            'pay_parameters.*.add_deduct.in' => 'Add/Deduct must be 0 (Deduct) or 1 (Add).',
            'pay_parameters.*.Lg_Id.required' => 'Ledger is required for each parameter.',
            'pay_parameters.*.Lg_Id.exists' => 'Selected Ledger does not exist.',
            'pay_parameters.*.percentage.numeric' => 'Percentage must be a number.',
            'pay_parameters.*.percentage.min' => 'Percentage must be at least 0.',
            'pay_parameters.*.percentage.max' => 'Percentage must not exceed 100.',
            'pay_parameters.*.percentage.regex' => 'Percentage must have up to 2 decimal places.',
            'pay_parameters.*.amount_per_day.numeric' => 'Amount per day must be a number.',
            'pay_parameters.*.amount_per_day.min' => 'Amount per day must be at least 0.',
            'pay_parameters.*.amount_per_day.max' => 'Amount per day must not exceed 999999.99.',
            'pay_parameters.*.amount_per_day.regex' => 'Amount per day must have up to 2 decimal places.',
            'pay_parameters.*.calculation_formula.string' => 'Calculation Formula must be a string.',
            'pay_parameters.*.calculation_formula.max' => 'Calculation Formula must not exceed 255 characters.'
        ];
    }

    /**
     * Check if exact duplicate exists - CHANGED: tend_work_id से emp_type_id
     */
    public static function isExactDuplicate($empTypeId, $effectiveDate, $payParameters, $excludeId = null): bool
    {
        $query = self::where('emp_type_id', $empTypeId) // CHANGED
            ->whereDate('effective_date', $effectiveDate)
            ->whereJsonContains('pay_parameters', $payParameters)
            ->whereNull('deleted_at');

        if ($excludeId) {
            $query->where('pay_rate_id', '!=', $excludeId);
        }

        return $query->exists();
    }

    /**
     * Get all pay rates for an employee type and date - CHANGED
     */
    public static function getEntriesForEmployeeType($empTypeId, $effectiveDate)
    {
        return self::with(['employeeType'])
            ->where('emp_type_id', $empTypeId) // CHANGED
            ->whereDate('effective_date', $effectiveDate)
            ->whereNull('deleted_at')
            ->orderBy('effective_date', 'desc')
            ->get();
    }

    /**
     * Transform individual parameter for display
     */
    public static function transformParameterItem($param): array
    {
        $addDeduct = $param['add_deduct'] ?? 0;
        $percentage = isset($param['percentage']) ? (float)$param['percentage'] : null;
        $amountPerDay = isset($param['amount_per_day']) ? (float)$param['amount_per_day'] : null;

        $applicableCriteriaName = $param['applicable_criteria_name'] ?? '';

        return [
            'payroll_id' => $param['payroll_id'] ?? null,
            'applicable_criteria_name' => $applicableCriteriaName,
            'add_deduct' => $addDeduct,
            'add_deduct_text' => self::ADD_DEDUCT_TEXTS[$addDeduct] ?? 'Unknown',
            'Lg_Id' => $param['Lg_Id'] ?? null,
            'percentage' => $percentage,
            'percentage_formatted' => $percentage !== null ? number_format($percentage, 2) . '%' : null,
            'amount_per_day' => $amountPerDay,
            'amount_per_day_formatted' => $amountPerDay !== null ? '₹' . number_format($amountPerDay, 2) : null,
            'effective_percentage' => $percentage !== null ?
                ($addDeduct == self::DEDUCT ? -$percentage : $percentage) : 0,
            'effective_amount_per_day' => $amountPerDay !== null ?
                ($addDeduct == self::DEDUCT ? -$amountPerDay : $amountPerDay) : 0,
            'calculation_formula' => $param['calculation_formula'] ?? null,
        ];
    }
}
