<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class PaymentVoucher extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'payment_vouchers';
    protected $primaryKey = 'Pay_Voucher';
    public $incrementing = true;
    public $timestamps = true;

    protected $fillable = [
        'Pay_Vou_No',
        'bill_challan_date',
        'Br_Id',
        'Lg_Id',
        'SL_Id',
        'Tend_Alloc',
        'Tend_Work_Id',
        'basic_amount',
        'gross_amount',
        'total_tax_amount',
        'net_amount',
        'bill_amount',
        'payment_status',
        'bill_no',
        'challan_no',
        'remarks',
        'tax_parameters',
        'tax_calculations'
        // नए fields यहाँ नहीं जोड़े गए
    ];

    protected $casts = [
        'bill_challan_date' => 'date',
        'basic_amount' => 'decimal:2',
        'gross_amount' => 'decimal:2',
        'total_tax_amount' => 'decimal:2',
        'net_amount' => 'decimal:2',
        'bill_amount' => 'decimal:2',
        'tax_parameters' => 'array',
        'tax_calculations' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime'
    ];

    protected $appends = [
        'formatted_bill_date',
        'formatted_bill_amount',
        'formatted_basic_amount',
        'formatted_gross_amount',
        'formatted_net_amount',
        'formatted_tax_amount',
        'branch_name',
        'ledger_name',
        'sub_ledger_name',
        'tender_allocation_code',
        'tender_work_name',
        'tender_work_code',
        'tax_summary',
        'total_payable'
        // नए appends यहाँ नहीं जोड़े गए
    ];

    // Boot method to generate Pay_Vou_No
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->Pay_Vou_No)) {
                $model->Pay_Vou_No = static::generateVoucherNumber();
            }

            // Set default payment status to draft if not provided
            if (empty($model->payment_status)) {
                $model->payment_status = 'draft';
            }
        });
    }

    // Generate unique voucher number
    public static function generateVoucherNumber()
    {
        $prefix = 'PV';
        $date = date('Ym');

        // Get the last voucher number for this month
        $lastVoucher = static::where('Pay_Vou_No', 'like', "{$prefix}{$date}%")
            ->orderBy('Pay_Vou_No', 'desc')
            ->first();

        if ($lastVoucher) {
            $lastNumber = intval(substr($lastVoucher->Pay_Vou_No, -4));
            $newNumber = str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        } else {
            $newNumber = '0001';
        }

        return "{$prefix}{$date}{$newNumber}";
    }

    // Validation rules for create - SIRF STATUS VALUES UPDATE
    public static function createRules()
    {
        return [
            'Pay_Vou_No' => 'nullable|string|max:50|unique:payment_vouchers,Pay_Vou_No',
            'bill_challan_date' => 'required|date',
            'Br_Id' => 'required|integer|exists:main_departments,id',
            'Lg_Id' => 'required|integer|exists:ledger_master,Lg_Id',
            'SL_Id' => 'required|integer|exists:subledg_master,SL_Id',
            'Tend_Alloc' => 'nullable|integer|exists:tendor_allocation,Tend_Alloc',
            'Tend_Work_Id' => 'nullable|string|max:50|exists:new_tender_works,tend_work_id',
            'basic_amount' => 'required|numeric|min:0|max:9999999999999.99',
            'gross_amount' => 'nullable|numeric|min:0|max:9999999999999.99',
            'total_tax_amount' => 'nullable|numeric|min:-9999999999999.99|max:9999999999999.99',
            'net_amount' => 'nullable|numeric|min:0|max:9999999999999.99',
            'bill_amount' => 'nullable|numeric|min:0|max:9999999999999.99',
            'payment_status' => 'sometimes|in:draft,pending,approved,paid,cancelled,quary,hod_app,hod_query,acco_app,acco_query', // UPDATED: नए status जोड़े
            'bill_no' => 'nullable|string|max:100',
            'challan_no' => 'nullable|string|max:100',
            'remarks' => 'nullable|string',
            'tax_parameters' => 'nullable|array',
            'tax_parameters.*.tax_type' => 'required_with:tax_parameters|string',
            'tax_parameters.*.percentage' => 'required_with:tax_parameters|numeric|min:0|max:100',
            'tax_parameters.*.add_deduct' => 'required_with:tax_parameters|in:0,1',
            'tax_parameters.*.formula' => 'nullable|string',
            'tax_parameters.*.ledger_id' => 'nullable|integer',
            'tax_calculations' => 'nullable|array'
        ];
    }

    // Validation rules for update - SIRF STATUS VALUES UPDATE
    public static function updateRules($id)
    {
        return [
            'Pay_Vou_No' => 'sometimes|nullable|string|max:50|unique:payment_vouchers,Pay_Vou_No,' . $id . ',Pay_Voucher',
            'bill_challan_date' => 'sometimes|required|date',
            'Br_Id' => 'sometimes|required|integer|exists:main_departments,id',
            'Lg_Id' => 'sometimes|required|integer|exists:ledger_master,Lg_Id',
            'SL_Id' => 'sometimes|required|integer|exists:subledg_master,SL_Id',
            'Tend_Alloc' => 'nullable|integer|exists:tendor_allocation,Tend_Alloc',
            'Tend_Work_Id' => 'nullable|string|max:50|exists:new_tender_works,tend_work_id',
            'basic_amount' => 'sometimes|required|numeric|min:0|max:9999999999999.99',
            'gross_amount' => 'nullable|numeric|min:0|max:9999999999999.99',
            'total_tax_amount' => 'nullable|numeric|min:-9999999999999.99|max:9999999999999.99',
            'net_amount' => 'nullable|numeric|min:0|max:9999999999999.99',
            'bill_amount' => 'sometimes|nullable|numeric|min:0|max:9999999999999.99',
            'payment_status' => 'sometimes|in:draft,pending,approved,paid,cancelled,quary,hod_app,hod_query,acco_app,acco_query', // UPDATED: नए status जोड़े
            'bill_no' => 'nullable|string|max:100',
            'challan_no' => 'nullable|string|max:100',
            'remarks' => 'nullable|string',
            'tax_parameters' => 'nullable|array',
            'tax_parameters.*.tax_type' => 'required_with:tax_parameters|string',
            'tax_parameters.*.percentage' => 'required_with:tax_parameters|numeric|min:0|max:100',
            'tax_parameters.*.add_deduct' => 'required_with:tax_parameters|in:0,1',
            'tax_parameters.*.formula' => 'nullable|string',
            'tax_parameters.*.ledger_id' => 'nullable|integer',
            'tax_calculations' => 'nullable|array'
        ];
    }

    // Custom validation messages
    public static function validationMessages()
    {
        return [
            'Pay_Vou_No.unique' => 'This voucher number already exists',
            'Pay_Vou_No.max' => 'Voucher number cannot exceed 50 characters',
            'bill_challan_date.required' => 'Bill/Challan date is required',
            'bill_challan_date.date' => 'Invalid date format',
            'Br_Id.required' => 'Branch is required',
            'Br_Id.exists' => 'Selected branch does not exist',
            'Lg_Id.required' => 'Ledger account is required',
            'Lg_Id.exists' => 'Selected ledger account does not exist',
            'SL_Id.required' => 'Vendor is required',
            'SL_Id.exists' => 'Selected vendor does not exist',
            'Tend_Alloc.exists' => 'Selected tender allocation does not exist',
            'Tend_Work_Id.exists' => 'Selected tender work does not exist',
            'basic_amount.required' => 'Basic amount is required',
            'basic_amount.numeric' => 'Basic amount must be a number',
            'basic_amount.min' => 'Basic amount cannot be negative',
            'gross_amount.numeric' => 'Gross amount must be a number',
            'total_tax_amount.numeric' => 'Total tax amount must be a number',
            'net_amount.numeric' => 'Net amount must be a number',
            'bill_amount.numeric' => 'Bill amount must be a number',
            'bill_amount.min' => 'Bill amount cannot be negative',
            'payment_status.in' => 'Invalid payment status',
            'tax_parameters.array' => 'Tax parameters must be an array',
            'tax_parameters.*.tax_type.required' => 'Tax type is required for each parameter',
            'tax_parameters.*.percentage.required' => 'Percentage is required for each parameter',
            'tax_parameters.*.add_deduct.required' => 'Add/Deduct type is required for each parameter',
            'tax_parameters.*.add_deduct.in' => 'Add/Deduct must be 0 (Deduct) or 1 (Add)',
            'tax_calculations.array' => 'Tax calculations must be an array'
        ];
    }

    // Relationships
    public function branch()
    {
        return $this->belongsTo(MainDepartment::class, 'Br_Id', 'id');
    }

    public function ledger()
    {
        return $this->belongsTo(LedgerMaster::class, 'Lg_Id', 'Lg_Id');
    }

    public function subLedger()
    {
        return $this->belongsTo(SubLedgerMaster::class, 'SL_Id', 'SL_Id');
    }

    public function tenderAllocation()
    {
        return $this->belongsTo(TendorAllocation::class, 'Tend_Alloc', 'Tend_Alloc');
    }

    public function tenderWork()
    {
        return $this->belongsTo(NewTenderWork::class, 'Tend_Work_Id', 'tend_work_id');
    }

    // Scopes
    public function scopeByVoucherNumber($query, $voucherNo)
    {
        return $query->where('Pay_Vou_No', 'like', "%{$voucherNo}%");
    }

    public function scopeByBranch($query, $branchId)
    {
        return $query->where('Br_Id', $branchId);
    }

    public function scopeByLedger($query, $ledgerId)
    {
        return $query->where('Lg_Id', $ledgerId);
    }

    public function scopeBySubLedger($query, $subLedgerId)
    {
        return $query->where('SL_Id', $subLedgerId);
    }

    public function scopeByTenderWork($query, $tenderWorkId)
    {
        return $query->where('Tend_Work_Id', $tenderWorkId);
    }

    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('bill_challan_date', [$startDate, $endDate]);
    }

    public function scopeByPaymentStatus($query, $status)
    {
        return $query->where('payment_status', $status);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('Pay_Vou_No', 'like', "%{$search}%")
              ->orWhere('bill_no', 'like', "%{$search}%")
              ->orWhere('challan_no', 'like', "%{$search}%")
              ->orWhere('remarks', 'like', "%{$search}%")
              ->orWhere('bill_amount', 'like', "%{$search}%")
              ->orWhere('basic_amount', 'like', "%{$search}%")
              ->orWhere('gross_amount', 'like', "%{$search}%")
              ->orWhere('net_amount', 'like', "%{$search}%")
              ->orWhereHas('branch', function ($q) use ($search) {
                  $q->where('desc', 'like', "%{$search}%")
                    ->orWhere('Grcod', 'like', "%{$search}%");
              })
              ->orWhereHas('ledger', function ($q) use ($search) {
                  $q->where('Lg_Code', 'like', "%{$search}%")
                    ->orWhere('Lg_Name', 'like', "%{$search}%");
              })
              ->orWhereHas('subLedger', function ($q) use ($search) {
                  $q->where('SL_Code', 'like', "%{$search}%")
                    ->orWhere('SL_Name', 'like', "%{$search}%");
              })
              ->orWhereHas('tenderAllocation', function ($q) use ($search) {
                  $q->where('Tend_Alloc', 'like', "%{$search}%");
              })
              ->orWhereHas('tenderWork', function ($q) use ($search) {
                  $q->where('work_code', 'like', "%{$search}%")
                    ->orWhere('work_name', 'like', "%{$search}%");
              });
        });
    }

    // Attribute accessors
    public function getFormattedBillDateAttribute()
    {
        return $this->bill_challan_date ? $this->bill_challan_date->format('d-m-Y') : null;
    }

    public function getFormattedBillAmountAttribute()
    {
        return $this->bill_amount ? number_format($this->bill_amount, 2) : '0.00';
    }

    public function getFormattedBasicAmountAttribute()
    {
        return $this->basic_amount ? number_format($this->basic_amount, 2) : '0.00';
    }

    public function getFormattedGrossAmountAttribute()
    {
        return $this->gross_amount ? number_format($this->gross_amount, 2) : '0.00';
    }

    public function getFormattedNetAmountAttribute()
    {
        return $this->net_amount ? number_format($this->net_amount, 2) : '0.00';
    }

    public function getFormattedTaxAmountAttribute()
    {
        return $this->total_tax_amount ? number_format($this->total_tax_amount, 2) : '0.00';
    }

    public function getBranchNameAttribute()
    {
        return $this->branch ? $this->branch->desc : null;
    }

    public function getLedgerNameAttribute()
    {
        return $this->ledger ? $this->ledger->Lg_Name : null;
    }

    public function getSubLedgerNameAttribute()
    {
        return $this->subLedger ? $this->subLedger->SL_Name : null;
    }

    public function getTenderAllocationCodeAttribute()
    {
        return $this->tenderAllocation ? $this->tenderAllocation->Tend_Alloc : null;
    }

    public function getTenderWorkNameAttribute()
    {
        return $this->tenderWork ? $this->tenderWork->work_name : null;
    }

    public function getTenderWorkCodeAttribute()
    {
        return $this->tenderWork ? $this->tenderWork->work_code : null;
    }

    public function getTotalPayableAttribute()
    {
        return $this->net_amount ?? $this->bill_amount ?? 0;
    }

    public function getTaxSummaryAttribute()
    {
        if (!is_array($this->tax_calculations)) {
            return [];
        }

        $summary = [];
        foreach ($this->tax_calculations as $calc) {
            $summary[] = [
                'tax_type' => $calc['tax_type'] ?? '',
                'percentage' => $calc['percentage'] ?? 0,
                'amount' => isset($calc['amount']) ? number_format($calc['amount'], 2) : '0.00',
                'formula' => $calc['formula'] ?? '',
                'description' => $calc['description'] ?? '',
                'type' => isset($calc['add_deduct']) && $calc['add_deduct'] == 1 ? 'addition' : 'deduction',
                'effective_amount' => isset($calc['effective_amount']) ? number_format($calc['effective_amount'], 2) : '0.00'
            ];
        }

        return $summary;
    }

    // Business logic methods - UPDATED: नए status जोड़े
    public function canBeDeleted()
    {
        // Only allow deletion if in draft, pending, or quary status
        // नए status (hod_query, acco_query) भी जोड़े
        return in_array($this->payment_status, ['draft', 'pending', 'quary', 'hod_query', 'acco_query']);
    }

  public function canBeEdited()
{
    $editableStatuses = ['draft', 'pending', 'quary', 'hod_query', 'acco_query', 'hod_app', 'acco_app'];
    return in_array($this->payment_status, $editableStatuses);
}

    public function getPaymentDetails()
    {
        return [
            'voucher_id' => $this->Pay_Voucher,
            'voucher_number' => $this->Pay_Vou_No,
            'bill_date' => $this->formatted_bill_date,
            'branch' => [
                'id' => $this->Br_Id,
                'name' => $this->branch_name
            ],
            'ledger_account' => [
                'id' => $this->Lg_Id,
                'name' => $this->ledger_name
            ],
            'vendor' => [
                'id' => $this->SL_Id,
                'name' => $this->sub_ledger_name
            ],
            'tender_allocation' => $this->tenderAllocation ? [
                'id' => $this->tenderAllocation->Tend_Alloc,
                'code' => $this->tender_allocation_code
            ] : null,
            'tender_work' => $this->tenderWork ? [
                'id' => $this->Tend_Work_Id,
                'code' => $this->tender_work_code,
                'name' => $this->tender_work_name
            ] : null,
            'basic_amount' => $this->basic_amount,
            'formatted_basic_amount' => $this->formatted_basic_amount,
            'gross_amount' => $this->gross_amount,
            'formatted_gross_amount' => $this->formatted_gross_amount,
            'total_tax_amount' => $this->total_tax_amount,
            'formatted_tax_amount' => $this->formatted_tax_amount,
            'net_amount' => $this->net_amount,
            'formatted_net_amount' => $this->formatted_net_amount,
            'bill_amount' => $this->bill_amount,
            'formatted_bill_amount' => $this->formatted_bill_amount,
            'payment_status' => $this->payment_status,
            'bill_no' => $this->bill_no,
            'challan_no' => $this->challan_no,
            'remarks' => $this->remarks,
            'tax_parameters' => $this->tax_parameters,
            'tax_calculations' => $this->tax_calculations,
            'tax_summary' => $this->tax_summary,
            'total_payable' => $this->total_payable,
            'formatted_total_payable' => number_format($this->total_payable, 2),
            'created_at' => $this->created_at ? $this->created_at->format('d-m-Y H:i:s') : null,
            'updated_at' => $this->updated_at ? $this->updated_at->format('d-m-Y H:i:s') : null
        ];
    }

   
    public function approve($approvedBy = null)
    {

        if (!in_array($this->payment_status, ['pending', 'quary', 'hod_app', 'acco_app'])) {
            return false;
        }

        $this->payment_status = 'approved';
        $this->approved_at = now();

        if ($approvedBy) {
            // If you have approved_by field
            // $this->approved_by = $approvedBy;
        }

        return $this->save();
    }

    /**
     * Mark as quary
     */
    public function markAsQuary()
    {
        $this->payment_status = 'quary';
        return $this->save();
    }

    /**
     * Mark as paid - UPDATED: approved से paid
     */
    public function markAsPaid()
    {
        if ($this->payment_status !== 'approved') {
            return false;
        }

        $this->payment_status = 'paid';
        $this->paid_at = now();

        return $this->save();
    }
}
