<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class RoleMaster extends Model
{
    use HasFactory;

    protected $table = 'role_master';
    protected $primaryKey = 'Role_Id';
    public $timestamps = true;

    protected $fillable = [
        'Role_Prefix',
        'Role_Name',
        'TwoFacAuth',
        'Is_Active'
    ];

    protected $casts = [
        'Is_Active' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Validation rules for create
    public static function createRules()
    {
        return [
            'Role_Prefix' => 'nullable|string|max:5',
            'Role_Name' => 'required|string|max:25|unique:role_master,Role_Name',
            'TwoFacAuth' => 'nullable|string|in:Y,N',
            'Is_Active' => 'nullable|boolean'
        ];
    }

    // Validation rules for update
    public static function updateRules($id)
    {
        return [
            'Role_Prefix' => 'nullable|string|max:5',
            'Role_Name' => 'sometimes|required|string|max:25|unique:role_master,Role_Name,' . $id . ',Role_Id',
            'TwoFacAuth' => 'nullable|string|in:Y,N',
            'Is_Active' => 'nullable|boolean'
        ];
    }

    // Custom validation messages
    public static function validationMessages()
    {
        return [
            'Role_Name.required' => 'Role Name is required',
            'Role_Name.max' => 'Role Name should not exceed 25 characters',
            'Role_Name.unique' => 'Role Name already exists',
            'Role_Prefix.max' => 'Role Prefix should not exceed 5 characters',
            'TwoFacAuth.in' => 'Two Factor Authentication must be either Y or N'
        ];
    }

    /**
     * Relationship with Users through UserRoleMapping
     */
    public function users()
    {
        return $this->belongsToMany(
            GmcUser::class,
            'user_role_mapping',
            'Role_Id',
            'U_Id',
            'Role_Id',
            'user_id'
        )->withPivot('Default_Rid', 'URoleMap', 'created_at', 'updated_at');
    }

    /**
     * Relationship with UserRoleMapping
     */
    public function userMappings()
    {
        return $this->hasMany(UserRoleMapping::class, 'Role_Id', 'Role_Id');
    }

    /**
     * Get users with this role as default
     */
    public function defaultUsers()
    {
        return $this->users()
            ->wherePivot('Default_Rid', 1)
            ->get();
    }

    /**
     * Get users with this role as non-default
     */
    public function nonDefaultUsers()
    {
        return $this->users()
            ->wherePivot('Default_Rid', 0)
            ->get();
    }

    /**
     * Count total users with this role
     */
    public function userCount()
    {
        return $this->users()->count();
    }

    /**
     * Count users with this role as default
     */
    public function defaultUserCount()
    {
        return $this->users()
            ->wherePivot('Default_Rid', 1)
            ->count();
    }

    /**
     * Check if role is assigned to specific user
     */
    public function isAssignedToUser($userId)
    {
        return $this->users()
            ->where('user_id', $userId)
            ->exists();
    }

    /**
     * Get role mapping for specific user
     */
    public function getUserMapping($userId)
    {
        return $this->userMappings()
            ->where('U_Id', $userId)
            ->first();
    }

    /**
     * Check if role is default for specific user
     */
    public function isDefaultForUser($userId)
    {
        $mapping = $this->getUserMapping($userId);
        return $mapping ? $mapping->Default_Rid === 1 : false;
    }

    /**
     * Get active roles only
     */
    public function scopeActive($query)
    {
        return $query->where('Is_Active', true);
    }

    /**
     * Get inactive roles only
     */
    public function scopeInactive($query)
    {
        return $query->where('Is_Active', false);
    }

    /**
     * Search roles by name, prefix or TwoFacAuth
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('Role_Name', 'like', "%{$search}%")
              ->orWhere('Role_Prefix', 'like', "%{$search}%")
              ->orWhere('TwoFacAuth', 'like', "%{$search}%");
        });
    }

    /**
     * Get roles with user count
     */
    public function scopeWithUserCount($query)
    {
        return $query->withCount('users');
    }

    /**
     * Get roles with default user count
     */
    public function scopeWithDefaultUserCount($query)
    {
        return $query->withCount(['users as default_users_count' => function ($query) {
            $query->where('default_rid', 1);
        }]);
    }

    /**
     * Get roles ordered by user count
     */
    public function scopeOrderByUserCount($query, $direction = 'desc')
    {
        return $query->withCount('users')->orderBy('users_count', $direction);
    }

    /**
     * Get roles with Two Factor Authentication enabled
     */
    public function scopeWithTwoFactorAuth($query)
    {
        return $query->where('TwoFacAuth', 'Y');
    }

    /**
     * Get roles without Two Factor Authentication
     */
    public function scopeWithoutTwoFactorAuth($query)
    {
        return $query->where('TwoFacAuth', 'N')->orWhereNull('TwoFacAuth');
    }

    /**
     * Assign role to user
     */
    public function assignToUser($userId, $isDefault = false)
    {
        // Check if already assigned
        if ($this->isAssignedToUser($userId)) {
            return false;
        }

        // If setting as default, remove default from other roles of this user
        if ($isDefault) {
            UserRoleMapping::where('U_Id', $userId)
                ->update(['Default_Rid' => 0]);
        }

        // Create mapping
        return UserRoleMapping::create([
            'U_Id' => $userId,
            'Role_Id' => $this->Role_Id,
            'Default_Rid' => $isDefault ? 1 : 0
        ]);
    }

    /**
     * Remove role from user
     */
    public function removeFromUser($userId)
    {
        $mapping = $this->getUserMapping($userId);

        if (!$mapping) {
            return false;
        }

        $wasDefault = $mapping->Default_Rid === 1;
        $result = $mapping->delete();

        // If default role was removed, set another role as default
        if ($wasDefault && $result) {
            $user = GmcUser::find($userId);
            if ($user) {
                $newDefault = $user->roleMappings()->first();
                if ($newDefault) {
                    $newDefault->update(['Default_Rid' => 1]);
                }
            }
        }

        return $result;
    }

    /**
     * Set role as default for user
     */
    public function setAsDefaultForUser($userId)
    {
        // Check if role is assigned to user
        if (!$this->isAssignedToUser($userId)) {
            return false;
        }

        // Remove default from all roles of this user
        UserRoleMapping::where('U_Id', $userId)
            ->update(['Default_Rid' => 0]);

        // Set this role as default
        return UserRoleMapping::where('U_Id', $userId)
            ->where('Role_Id', $this->Role_Id)
            ->update(['Default_Rid' => 1]);
    }

    /**
     * Get all permissions for this role
     * (Assuming you have a permissions table)
     */
    public function permissions()
    {
        // If you have a permissions table, add this relationship
        // return $this->belongsToMany(Permission::class, 'role_permissions', 'role_id', 'permission_id');
    }

    /**
     * Check if role has specific permission
     */
    public function hasPermission($permission)
    {
        // Implement based on your permission system
        return true; // Placeholder
    }

    /**
     * Get role hierarchy level
     * (Assuming you have hierarchy field)
     */
    public function getHierarchyLevelAttribute()
    {
        // Example: SUPER_ADMIN = 1, ADMIN = 2, DOCTOR = 3, etc.
        $hierarchy = [
            'SUPER_ADMIN' => 1,
            'ADMIN' => 2,
            'DOCTOR' => 3,
            'NURSE' => 4,
            'STAFF' => 5
        ];

        return $hierarchy[$this->Role_Prefix] ?? 999;
    }

    /**
     * Check if this role can manage another role
     */
    public function canManage(RoleMaster $otherRole)
    {
        return $this->hierarchy_level <= $otherRole->hierarchy_level;
    }

    /**
     * Get all manageable roles
     */
    public function manageableRoles()
    {
        return self::where('Is_Active', true)
            ->get()
            ->filter(function ($role) {
                return $this->canManage($role);
            });
    }

    /**
     * Bulk assign role to multiple users
     */
    public function bulkAssignToUsers(array $userIds, $isDefault = false)
    {
        $results = [
            'assigned' => [],
            'failed' => [],
            'skipped' => []
        ];

        foreach ($userIds as $userId) {
            try {
                // Check if already assigned
                if ($this->isAssignedToUser($userId)) {
                    $results['skipped'][] = $userId;
                    continue;
                }

                // If setting as default, remove default from other roles
                if ($isDefault) {
                    UserRoleMapping::where('U_Id', $userId)
                        ->update(['Default_Rid' => 0]);
                }

                // Create mapping
                UserRoleMapping::create([
                    'U_Id' => $userId,
                    'Role_Id' => $this->Role_Id,
                    'Default_Rid' => $isDefault ? 1 : 0
                ]);

                $results['assigned'][] = $userId;

            } catch (\Exception $e) {
                $results['failed'][] = [
                    'user_id' => $userId,
                    'error' => $e->getMessage()
                ];
            }
        }

        return $results;
    }

    /**
     * Bulk remove role from multiple users
     */
    public function bulkRemoveFromUsers(array $userIds)
    {
        $results = [
            'removed' => [],
            'failed' => []
        ];

        foreach ($userIds as $userId) {
            try {
                $success = $this->removeFromUser($userId);

                if ($success) {
                    $results['removed'][] = $userId;
                } else {
                    $results['failed'][] = [
                        'user_id' => $userId,
                        'error' => 'Role not assigned to user'
                    ];
                }

            } catch (\Exception $e) {
                $results['failed'][] = [
                    'user_id' => $userId,
                    'error' => $e->getMessage()
                ];
            }
        }

        return $results;
    }

    // Attribute accessors
    public function getTwoFactorAuthTextAttribute()
    {
        return $this->TwoFacAuth == 'Y' ? 'Enabled' : ($this->TwoFacAuth == 'N' ? 'Disabled' : 'Not Set');
    }

    public function getStatusTextAttribute()
    {
        return $this->Is_Active ? 'Active' : 'Inactive';
    }

    public function getFullRoleAttribute()
    {
        $parts = [];
        if ($this->Role_Prefix) {
            $parts[] = $this->Role_Prefix;
        }
        $parts[] = $this->Role_Name;
        return implode(' - ', $parts);
    }

    // Business logic methods
    public function canBeDeleted()
    {
        // Check if role has users assigned
        return $this->users()->count() === 0;
    }

    public function deactivate()
    {
        $this->update(['Is_Active' => false]);
        return $this;
    }

    public function activate()
    {
        $this->update(['Is_Active' => true]);
        return $this;
    }

    public function enableTwoFactorAuth()
    {
        $this->update(['TwoFacAuth' => 'Y']);
        return $this;
    }

    public function disableTwoFactorAuth()
    {
        $this->update(['TwoFacAuth' => 'N']);
        return $this;
    }
}
