<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class VendorInvoiceParameter extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'vendor_invoice_parameters';
    protected $primaryKey = 'proposal_id';
    public $incrementing = true;
    protected $keyType = 'int';

    protected $fillable = [
        'tend_work_id',
        'effective_date',
        'tax_parameters'
    ];

    protected $casts = [
        'effective_date' => 'date',
        'tax_parameters' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime'
    ];

    const DEDUCT = 0;
    const ADD = 1;



    const ADD_DEDUCT_TEXTS = [
        self::DEDUCT => 'Deduct',
        self::ADD => 'Add'
    ];

   

    protected $appends = [
        'total_parameters',
        'total_percentage',
        'is_active'
    ];

    /**
     * Get the tender work - CORRECTED
     * tend_work_id (vendor_invoice_parameters) references tend_work_id (new_tender_works)
     */
    public function tenderWork()
    {
        return $this->belongsTo(NewTenderWork::class, 'tend_work_id', 'tend_work_id');
    }

    /**
     * Get total number of parameters
     */
    public function getTotalParametersAttribute(): int
    {
        return is_array($this->tax_parameters) ? count($this->tax_parameters) : 0;
    }

    /**
     * Calculate total effective percentage
     */
    public function getTotalPercentageAttribute(): float
    {
        if (!is_array($this->tax_parameters)) {
            return 0;
        }

        $total = 0;
        foreach ($this->tax_parameters as $param) {
            $percentage = $param['percentage'] ?? 0;
            $addDeduct = $param['add_deduct'] ?? self::DEDUCT;

            $effectivePercentage = ($addDeduct == self::DEDUCT) ? -$percentage : $percentage;
            $total += $effectivePercentage;
        }

        return $total;
    }

    /**
     * Check if parameter is active
     */
    public function getIsActiveAttribute(): bool
    {
        return is_null($this->deleted_at);
    }

    /**
     * Calculate total amount based on base amount
     */
    public function calculateTotalAmount($baseAmount): float
    {
        if (!is_array($this->tax_parameters)) {
            return 0;
        }

        $total = 0;
        foreach ($this->tax_parameters as $param) {
            $percentage = $param['percentage'] ?? 0;
            if ($percentage > 0) {
                $total += ($baseAmount * $percentage) / 100;
            }
        }

        return $total;
    }

    /**
     * Validation rules for single entry with multiple parameters
     */
    public static function createRules()
    {
        return [
            'tend_work_id' => 'required|exists:new_tender_works,tend_work_id',
            'effective_date' => 'required|date|date_format:Y-m-d',
            'tax_parameters' => 'required|array|min:1',
            'tax_parameters.*.tax_type_id' => 'required|string|max:20|exists:fa_tax_type_master,tax_type_id',
            'tax_parameters.*.add_deduct' => 'required|in:0,1',
            'tax_parameters.*.Lg_Id' => 'required|exists:ledger_master,Lg_Id',
            'tax_parameters.*.percentage' => 'nullable|numeric|min:0|max:999999.99|regex:/^\d+(\.\d{1,2})?$/',
            'tax_parameters.*.tax_code' => 'nullable|string|max:20',
            // CHANGED: Replaced Cal_Flage with calculation_formula
            'tax_parameters.*.calculation_formula' => 'nullable|string|max:255'
        ];
    }

    /**
     * Custom validation messages
     */
    public static function validationMessages()
    {
        return [
            'tend_work_id.required' => 'Tender Work is required.',
            'tend_work_id.exists' => 'Selected Tender Work does not exist.',
            'effective_date.required' => 'Effective Date is required.',
            'effective_date.date' => 'Effective Date must be a valid date.',
            'effective_date.date_format' => 'Effective Date must be in YYYY-MM-DD format.',
            'tax_parameters.required' => 'At least one tax parameter is required.',
            'tax_parameters.*.tax_type_id.required' => 'Tax Type ID is required for each parameter.',
            'tax_parameters.*.tax_type_id.string' => 'Tax Type ID must be a string.',
            'tax_parameters.*.tax_type_id.max' => 'Tax Type ID must not exceed 20 characters.',
            'tax_parameters.*.tax_type_id.exists' => 'Selected Tax Type does not exist.',
            'tax_parameters.*.add_deduct.required' => 'Add/Deduct type is required for each parameter.',
            'tax_parameters.*.add_deduct.in' => 'Add/Deduct must be 0 (Deduct) or 1 (Add).',
            'tax_parameters.*.Lg_Id.required' => 'Ledger is required for each parameter.',
            'tax_parameters.*.Lg_Id.exists' => 'Selected Ledger does not exist.',
            'tax_parameters.*.percentage.numeric' => 'Percentage must be a number.',
            'tax_parameters.*.percentage.min' => 'Percentage must be at least 0.',
            'tax_parameters.*.percentage.max' => 'Percentage must not exceed 999999.99.',
            'tax_parameters.*.percentage.regex' => 'Percentage must have up to 2 decimal places.',
            'tax_parameters.*.tax_code.string' => 'Tax Code must be a string.',
            'tax_parameters.*.tax_code.max' => 'Tax Code must not exceed 20 characters.',
            // CHANGED: Updated validation message for calculation_formula
            'tax_parameters.*.calculation_formula.string' => 'Calculation Formula must be a string.',
            'tax_parameters.*.calculation_formula.max' => 'Calculation Formula must not exceed 255 characters.'
        ];
    }

    /**
     * Check if exact duplicate exists (same tender work, date and all parameters)
     */
    public static function isExactDuplicate($tendWorkId, $effectiveDate, $taxParameters, $excludeId = null): bool
    {
        $query = self::where('tend_work_id', $tendWorkId)
            ->whereDate('effective_date', $effectiveDate)
            ->whereJsonContains('tax_parameters', $taxParameters)
            ->whereNull('deleted_at');

        if ($excludeId) {
            $query->where('proposal_id', '!=', $excludeId);
        }

        return $query->exists();
    }

    /**
     * Get all parameters for a tender work and date
     */
    public static function getEntriesForTenderWork($tendWorkId, $effectiveDate)
    {
        return self::with(['tenderWork'])
            ->where('tend_work_id', $tendWorkId)
            ->whereDate('effective_date', $effectiveDate)
            ->whereNull('deleted_at')
            ->orderBy('effective_date', 'desc')
            ->get();
    }

    /**
     * Transform individual parameter for display
     */
    public static function transformParameterItem($param): array
    {
        $addDeduct = $param['add_deduct'] ?? 0;
        $percentage = isset($param['percentage']) ? (float)$param['percentage'] : null;

        return [
            'tax_type_id' => $param['tax_type_id'] ?? null,
            'tax_code' => $param['tax_code'] ?? null,
            'add_deduct' => $addDeduct,
            'add_deduct_text' => self::ADD_DEDUCT_TEXTS[$addDeduct] ?? 'Unknown',
            'Lg_Id' => $param['Lg_Id'] ?? null,
            'percentage' => $percentage,
            'percentage_formatted' => $percentage !== null ? number_format($percentage, 2) . '%' : null,
            'effective_percentage' => $percentage !== null ?
                ($addDeduct == self::DEDUCT ? -$percentage : $percentage) : 0,
            // CHANGED: Replaced Cal_Flage with calculation_formula
            'calculation_formula' => $param['calculation_formula'] ?? null,
            // REMOVED: Cal_Flage_text field
        ];
    }
}
