<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class WardStatistic extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'ward_statistics';
    protected $primaryKey = 'WD_St_Id';

    protected $fillable = [
        'Ward_Id',
        'St_Date',
        'Population',
        'Area_SqKm',
        'is_active'
    ];

    protected $casts = [
        'WD_St_Id' => 'integer',
        'Ward_Id' => 'integer',
        'Area_SqKm' => 'decimal:2',
        'is_active' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Validation rules for create
    public static function createRules()
    {
        return [
            'Ward_Id' => 'required|exists:ward_master,Ward_Id',
            'St_Date' => 'required|string|max:25',
            'Population' => 'nullable|string|max:1',
            'Area_SqKm' => 'nullable|numeric|min:0|max:99999999.99',
        ];
    }

    // Validation rules for update
    public static function updateRules($id)
    {
        return [
            'Ward_Id' => 'sometimes|required|exists:ward_master,Ward_Id',
            'St_Date' => 'sometimes|required|string|max:25',
            'Population' => 'nullable|string|max:1',
            'Area_SqKm' => 'nullable|numeric|min:0|max:99999999.99',
            'is_active' => 'sometimes|boolean'
        ];
    }

    // Custom validation messages
    public static function validationMessages()
    {
        return [
            'Ward_Id.required' => 'Ward is required',
            'Ward_Id.exists' => 'Selected ward does not exist',
            'St_Date.required' => 'Statistics date is required',
            'St_Date.max' => 'Date cannot exceed 25 characters',
            'St_Date.unique' => 'Statistics already exist for this ward on this date',
            'Population.max' => 'Population code cannot exceed 1 character',
            'Area_SqKm.numeric' => 'Area must be a number',
            'Area_SqKm.min' => 'Area cannot be negative',
            'Area_SqKm.max' => 'Area value is too large'
        ];
    }

    // Custom validation for unique statistics per ward and date
    public static function validateUniqueStatistic($wardId, $stDate, $excludeId = null)
    {
        $query = self::where('Ward_Id', $wardId)
                    ->where('St_Date', $stDate);

        if ($excludeId) {
            $query->where('WD_St_Id', '!=', $excludeId);
        }

        return !$query->exists();
    }

    // Relationships
    public function ward()
    {
        return $this->belongsTo(WardMaster::class, 'Ward_Id', 'Ward_Id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByWard($query, $wardId)
    {
        return $query->where('Ward_Id', $wardId);
    }

    public function scopeByDate($query, $date)
    {
        return $query->where('St_Date', $date);
    }

    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('St_Date', [$startDate, $endDate]);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('St_Date', 'like', "%{$search}%")
              ->orWhere('Population', 'like', "%{$search}%")
              ->orWhereHas('ward', function ($q) use ($search) {
                  $q->where('Ward_Code', 'like', "%{$search}%")
                    ->orWhere('Ward_Name', 'like', "%{$search}%")
                    ->orWhereHas('zone', function ($q) use ($search) {
                        $q->where('Zone_Code', 'like', "%{$search}%")
                          ->orWhere('Zone_Name', 'like', "%{$search}%");
                    });
              });
        });
    }

    public function scopeLatestFirst($query)
    {
        return $query->orderBy('St_Date', 'desc')->orderBy('created_at', 'desc');
    }

    // Attribute accessors
    public function getWardNameAttribute()
    {
        return $this->ward ? $this->ward->Ward_Name : null;
    }

    public function getWardCodeAttribute()
    {
        return $this->ward ? $this->ward->Ward_Code : null;
    }

    public function getZoneNameAttribute()
    {
        return $this->ward && $this->ward->zone ? $this->ward->zone->Zone_Name : null;
    }

    public function getZoneCodeAttribute()
    {
        return $this->ward && $this->ward->zone ? $this->ward->zone->Zone_Code : null;
    }

    public function getFormattedAreaAttribute()
    {
        return $this->Area_SqKm ? number_format($this->Area_SqKm, 2) . ' sq km' : 'N/A';
    }

    public function getPopulationLabelAttribute()
    {
        $populationLabels = [
            'H' => 'High',
            'M' => 'Medium',
            'L' => 'Low',
            'V' => 'Very High',
            'X' => 'Very Low'
        ];

        return $populationLabels[$this->Population] ?? 'Unknown';
    }

    // Business logic methods
    public function canBeDeleted()
    {
        // Add logic if there are any dependencies
        return true;
    }

    public function deactivate()
    {
        $this->update(['is_active' => false]);
        return $this;
    }

    public function activate()
    {
        $this->update(['is_active' => true]);
        return $this;
    }

    // Helper methods for date format
    public function getDateAttribute()
    {
        // You can add date formatting logic here
        return $this->St_Date;
    }
}
