<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class ZoneMaster extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'zone_masters';
    protected $primaryKey = 'Zone_Id';

    protected $fillable = [
        'Zone_Code',
        'Zone_Name',
        'Org_Id'
    ];

    protected $casts = [
        'Zone_Id' => 'integer',
        'Org_Id' => 'integer'
    ];

    // Validation rules for create
    public static function createRules()
    {
        return [
            'Zone_Code' => 'required|string|max:5',
            'Zone_Name' => 'required|string|max:25',
            'Org_Id' => 'nullable|integer|exists:organisation_settings,Org_Id'
        ];
    }

    // Validation rules for update
    public static function updateRules($id)
    {
        return [
            'Zone_Code' => 'sometimes|required|string|max:5',
            'Zone_Name' => 'sometimes|required|string|max:25',
            'Org_Id' => 'nullable|integer|exists:organisation_settings,Org_Id'
        ];
    }

    // Custom validation messages
    public static function validationMessages()
    {
        return [
            'Zone_Code.required' => 'Zone code is required',
            'Zone_Code.max' => 'Zone code cannot exceed 5 characters',
            'Zone_Code.unique' => 'This zone code already exists for the selected organization',
            'Zone_Name.required' => 'Zone name is required',
            'Zone_Name.max' => 'Zone name cannot exceed 25 characters',
            'Org_Id.exists' => 'The selected organization does not exist'
        ];
    }

    // Custom validation for unique Zone_Code within organization
    public static function validateUniqueZoneCode($zoneCode, $orgId, $excludeId = null)
    {
        $query = self::where('Zone_Code', $zoneCode);

        if ($orgId === null) {
            $query->whereNull('Org_Id');
        } else {
            $query->where('Org_Id', $orgId);
        }

        if ($excludeId) {
            $query->where('Zone_Id', '!=', $excludeId);
        }

        return !$query->exists();
    }

    // Relationships
    public function organisation()
    {
        return $this->belongsTo(OrganisationSetting::class, 'Org_Id', 'Org_Id');
    }

    // Scopes
    public function scopeByOrganization($query, $orgId)
    {
        if ($orgId === null) {
            return $query->whereNull('Org_Id');
        }
        return $query->where('Org_Id', $orgId);
    }

    public function scopeWithoutOrganization($query)
    {
        return $query->whereNull('Org_Id');
    }

    public function scopeWithOrganization($query)
    {
        return $query->whereNotNull('Org_Id');
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('Zone_Code', 'like', "%{$search}%")
              ->orWhere('Zone_Name', 'like', "%{$search}%")
              ->orWhereHas('organisation', function ($q) use ($search) {
                  $q->where('Org_Code', 'like', "%{$search}%")
                    ->orWhere('Org_Name', 'like', "%{$search}%");
              });
        });
    }

    // Attribute accessors
    public function getFullZoneAttribute()
    {
        return $this->Zone_Code . ' - ' . $this->Zone_Name;
    }

    public function getOrganizationNameAttribute()
    {
        return $this->organisation ? $this->organisation->Org_Name : 'Global';
    }

    public function getOrganizationCodeAttribute()
    {
        return $this->organisation ? $this->organisation->Org_Code : null;
    }

    // Check if zone is global (no organization)
    public function isGlobal()
    {
        return $this->Org_Id === null;
    }

    // Business logic methods
    public function canBeDeleted()
    {
        // Add logic to check if zone has dependencies
        // Example: check if there are branches in this zone
        return true;
    }
}
