<?php

namespace App\Services;

use Illuminate\Support\Facades\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class DeviceDetectionService
{
    /**
     * Get all device information automatically
     */
    public static function getDeviceInfo($userId = null)
    {
        try {
            $info = [
                'device_name' => self::getDeviceName(),
                'device_type' => self::getDeviceType(),
                'device_category' => self::getDeviceCategory(),
                'ip_address' => self::getIpAddress(),
                'mac_address' => self::getMacAddress(),
                'hdd_serial' => self::getHddSerial(),
                'motherboard_serial' => self::getMotherboardSerial(),
                'processor_id' => self::getProcessorId(),
                'operating_system' => self::getOperatingSystem(),
                'browser' => self::getBrowser(),
                'browser_fingerprint' => self::getBrowserFingerprint(),
                'device_fingerprint' => self::generateDeviceFingerprint(),
                'user_agent' => request()->header('User-Agent'),
                'screen_resolution' => self::getScreenResolution(),
                'timezone' => self::getTimezone(),
                'plugins' => self::getBrowserPlugins(),
                'fonts' => self::getSystemFonts(),
                'canvas_fingerprint' => self::getCanvasFingerprint(),
                'webgl_fingerprint' => self::getWebGLFingerprint(),
                'cpu_cores' => self::getCPUCores(),
                'ram_size' => self::getRAMSize(),
                'gpu_info' => self::getGPUInfo(),
                'detected_at' => now()->toDateTimeString()
            ];

            // Log successful detection
            Log::info('Device detected for user: ' . ($userId ?? 'unknown'), $info);

            return $info;

        } catch (\Exception $e) {
            Log::error('Device detection failed: ' . $e->getMessage());

            // Return basic info if detection fails
            return [
                'device_name' => 'Unknown Device',
                'device_type' => 'UNKNOWN',
                'device_category' => 'SECONDARY',
                'ip_address' => request()->ip(),
                'mac_address' => '00:00:00:00:00:00',
                'hdd_serial' => 'UNKNOWN-HDD',
                'operating_system' => php_uname('s'),
                'browser' => request()->header('User-Agent'),
                'browser_fingerprint' => 'fp_error_' . Str::random(10),
                'detected_at' => now()->toDateTimeString()
            ];
        }
    }

    /**
     * Get device name (combination of hostname + user agent)
     */
    private static function getDeviceName()
    {
        $hostname = gethostname() ?: 'Unknown-Host';
        $browser = self::getBrowser();
        $os = self::getOperatingSystem();

        return $hostname . '-' . $os . '-' . $browser;
    }

    /**
     * Determine device type based on user agent
     */
    private static function getDeviceType()
    {
        $userAgent = strtolower(request()->header('User-Agent', ''));

        if (strpos($userAgent, 'mobile') !== false ||
            strpos($userAgent, 'android') !== false ||
            strpos($userAgent, 'iphone') !== false) {
            return 'MOBILE';
        } elseif (strpos($userAgent, 'tablet') !== false ||
                 strpos($userAgent, 'ipad') !== false) {
            return 'TABLET';
        } elseif (strpos($userAgent, 'windows') !== false ||
                 strpos($userAgent, 'linux') !== false ||
                 strpos($userAgent, 'mac') !== false) {
            return 'DESKTOP';
        } elseif (strpos($userAgent, 'bot') !== false ||
                 strpos($userAgent, 'crawler') !== false) {
            return 'BOT';
        } else {
            return 'UNKNOWN';
        }
    }

    /**
     * Determine device category (for now always PRIMARY)
     */
    private static function getDeviceCategory()
    {
        return 'PRIMARY'; // First device is always primary
    }

    /**
     * Get IP address
     */
    private static function getIpAddress()
    {
        return request()->ip();
    }

    /**
     * Get MAC address (server-side attempt)
     */
    private static function getMacAddress()
    {
        try {
            // Method 1: Get MAC via system command (Linux)
            if (strtoupper(substr(PHP_OS, 0, 3)) !== 'WIN') {
                $mac = exec('cat /sys/class/net/eth0/address 2>/dev/null');
                if ($mac) return $mac;

                $mac = exec('cat /sys/class/net/wlan0/address 2>/dev/null');
                if ($mac) return $mac;
            }

            // Method 2: Get MAC via arp (cross-platform)
            $ip = request()->ip();
            if (filter_var($ip, FILTER_VALIDATE_IP)) {
                $arp = @shell_exec("arp -a " . escapeshellarg($ip));
                if ($arp) {
                    preg_match('/([a-fA-F0-9]{2}[:-]){5}[a-fA-F0-9]{2}/', $arp, $matches);
                    if (!empty($matches[0])) return $matches[0];
                }
            }

            // Method 3: Try to get MAC via PHP exec
            if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                $output = [];
                exec("getmac /FO CSV /NH", $output);
                if (!empty($output[0])) {
                    $parts = str_getcsv($output[0]);
                    if (!empty($parts[0])) return $parts[0];
                }
            }

            // Return default if all methods fail
            return '00:00:00:00:00:00';

        } catch (\Exception $e) {
            return '00:00:00:00:00:00';
        }
    }

    /**
     * Get HDD Serial (server-side attempt)
     */
    private static function getHddSerial()
    {
        try {
            if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                // Windows
                $output = shell_exec('wmic diskdrive get serialnumber 2>nul');
                if ($output) {
                    $lines = explode("\n", $output);
                    foreach ($lines as $line) {
                        $serial = trim($line);
                        if (!empty($serial) && $serial !== 'SerialNumber') {
                            return $serial;
                        }
                    }
                }
            } else {
                // Linux/Mac
                $output = shell_exec('udevadm info --query=property --name=/dev/sda 2>/dev/null | grep ID_SERIAL_SHORT');
                if ($output) {
                    $parts = explode('=', $output);
                    if (!empty($parts[1])) return trim($parts[1]);
                }

                // Alternative for Linux
                $output = shell_exec('lsblk -o serial /dev/sda 2>/dev/null | tail -1');
                if ($output) return trim($output);
            }

            return 'UNKNOWN-HDD-' . Str::random(10);

        } catch (\Exception $e) {
            return 'UNKNOWN-HDD-' . Str::random(10);
        }
    }

    /**
     * Get Motherboard Serial
     */
    private static function getMotherboardSerial()
    {
        try {
            if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                $output = shell_exec('wmic baseboard get serialnumber 2>nul');
                if ($output) {
                    $lines = explode("\n", $output);
                    foreach ($lines as $line) {
                        $serial = trim($line);
                        if (!empty($serial) && $serial !== 'SerialNumber') {
                            return $serial;
                        }
                    }
                }
            } else {
                $output = shell_exec('sudo dmidecode -s baseboard-serial-number 2>/dev/null');
                if ($output) return trim($output);
            }

            return 'UNKNOWN-MB-' . Str::random(8);

        } catch (\Exception $e) {
            return 'UNKNOWN-MB-' . Str::random(8);
        }
    }

    /**
     * Get Processor ID
     */
    private static function getProcessorId()
    {
        try {
            if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                $output = shell_exec('wmic cpu get processorid 2>nul');
                if ($output) {
                    $lines = explode("\n", $output);
                    foreach ($lines as $line) {
                        $pid = trim($line);
                        if (!empty($pid) && $pid !== 'ProcessorId') {
                            return $pid;
                        }
                    }
                }
            } else {
                $output = shell_exec('sudo dmidecode -t processor 2>/dev/null | grep ID');
                if ($output) {
                    $lines = explode("\n", $output);
                    if (!empty($lines[0])) {
                        $parts = explode(':', $lines[0]);
                        if (!empty($parts[1])) return trim($parts[1]);
                    }
                }
            }

            return 'CPU-' . php_uname('m') . '-' . Str::random(6);

        } catch (\Exception $e) {
            return 'CPU-' . php_uname('m') . '-' . Str::random(6);
        }
    }

    /**
     * Get Operating System
     */
    private static function getOperatingSystem()
    {
        $userAgent = request()->header('User-Agent', '');

        if (strpos($userAgent, 'Windows') !== false) {
            if (strpos($userAgent, 'Windows NT 10.0') !== false) return 'Windows 10';
            if (strpos($userAgent, 'Windows NT 6.3') !== false) return 'Windows 8.1';
            if (strpos($userAgent, 'Windows NT 6.2') !== false) return 'Windows 8';
            if (strpos($userAgent, 'Windows NT 6.1') !== false) return 'Windows 7';
            return 'Windows';
        } elseif (strpos($userAgent, 'Mac') !== false) {
            return 'macOS';
        } elseif (strpos($userAgent, 'Linux') !== false) {
            return 'Linux';
        } elseif (strpos($userAgent, 'Android') !== false) {
            return 'Android';
        } elseif (strpos($userAgent, 'iOS') !== false || strpos($userAgent, 'iPhone') !== false) {
            return 'iOS';
        } else {
            return php_uname('s') . ' ' . php_uname('r');
        }
    }

    /**
     * Get Browser information
     */
    private static function getBrowser()
    {
        $userAgent = request()->header('User-Agent', '');

        if (strpos($userAgent, 'MSIE') !== false || strpos($userAgent, 'Trident') !== false) {
            return 'Internet Explorer';
        } elseif (strpos($userAgent, 'Edge') !== false) {
            return 'Microsoft Edge';
        } elseif (strpos($userAgent, 'Chrome') !== false) {
            // Extract Chrome version
            preg_match('/Chrome\/([0-9.]+)/', $userAgent, $matches);
            return 'Chrome' . (!empty($matches[1]) ? ' ' . $matches[1] : '');
        } elseif (strpos($userAgent, 'Firefox') !== false) {
            preg_match('/Firefox\/([0-9.]+)/', $userAgent, $matches);
            return 'Firefox' . (!empty($matches[1]) ? ' ' . $matches[1] : '');
        } elseif (strpos($userAgent, 'Safari') !== false) {
            return 'Safari';
        } elseif (strpos($userAgent, 'Opera') !== false || strpos($userAgent, 'OPR') !== false) {
            return 'Opera';
        } else {
            return 'Unknown Browser';
        }
    }

    /**
     * Generate browser fingerprint
     */
    private static function getBrowserFingerprint()
    {
        $components = [
            'user_agent' => request()->header('User-Agent', ''),
            'accept_language' => request()->header('Accept-Language', ''),
            'accept_encoding' => request()->header('Accept-Encoding', ''),
            'screen_resolution' => self::getScreenResolution(),
            'timezone' => self::getTimezone(),
            'platform' => php_uname('s'),
            'plugins' => self::getBrowserPlugins(),
            'fonts' => self::getSystemFonts(),
            'canvas' => self::getCanvasFingerprint(),
            'webgl' => self::getWebGLFingerprint(),
            'cpu_cores' => self::getCPUCores(),
            'timestamp' => time()
        ];

        $fingerprintString = json_encode($components);
        return 'fp_' . hash('sha256', $fingerprintString);
    }

    /**
     * Generate device fingerprint from all hardware info
     */
    private static function generateDeviceFingerprint()
    {
        $components = [
            'mac' => self::getMacAddress(),
            'hdd' => self::getHddSerial(),
            'mb' => self::getMotherboardSerial(),
            'cpu' => self::getProcessorId(),
            'os' => self::getOperatingSystem(),
            'browser' => self::getBrowser(),
            'ip' => self::getIpAddress()
        ];

        return 'device_' . hash('sha256', json_encode($components));
    }

    /**
     * Get screen resolution (from JavaScript via request)
     */
    private static function getScreenResolution()
    {
        return request()->header('X-Screen-Resolution', 'Unknown');
    }

    /**
     * Get timezone (from JavaScript via request)
     */
    private static function getTimezone()
    {
        return request()->header('X-Timezone', date_default_timezone_get());
    }

    /**
     * Get browser plugins (from JavaScript via request)
     */
    private static function getBrowserPlugins()
    {
        return request()->header('X-Browser-Plugins', '[]');
    }

    /**
     * Get system fonts (from JavaScript via request)
     */
    private static function getSystemFonts()
    {
        return request()->header('X-System-Fonts', '[]');
    }

    /**
     * Get canvas fingerprint (from JavaScript via request)
     */
    private static function getCanvasFingerprint()
    {
        return request()->header('X-Canvas-Fingerprint', '');
    }

    /**
     * Get WebGL fingerprint (from JavaScript via request)
     */
    private static function getWebGLFingerprint()
    {
        return request()->header('X-WebGL-Fingerprint', '');
    }

    /**
     * Get CPU cores (from JavaScript via request)
     */
    private static function getCPUCores()
    {
        return (int) request()->header('X-CPU-Cores', 1);
    }

    /**
     * Get RAM size (from JavaScript via request)
     */
    private static function getRAMSize()
    {
        return request()->header('X-RAM-Size', 'Unknown');
    }

    /**
     * Get GPU info (from JavaScript via request)
     */
    private static function getGPUInfo()
    {
        return request()->header('X-GPU-Info', 'Unknown');
    }

    /**
     * Check if device is on hospital network
     */
    public static function isOnHospitalNetwork($ipAddress = null)
    {
        $ip = $ipAddress ?: request()->ip();

        // GMC Gandhinagar IP ranges (example)
        $hospitalRanges = [
            '192.168.0.0/16',    // Internal network
            '10.0.0.0/8',        // Hospital internal
            '172.16.0.0/12',     // Hospital VPN
        ];

        foreach ($hospitalRanges as $range) {
            if (self::ipInRange($ip, $range)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if IP is in range
     */
    private static function ipInRange($ip, $range)
    {
        list($subnet, $bits) = explode('/', $range);
        $ip = ip2long($ip);
        $subnet = ip2long($subnet);
        $mask = -1 << (32 - $bits);
        $subnet &= $mask;

        return ($ip & $mask) == $subnet;
    }
}
