<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // First check the data type of tend_work_id in new_tender_works
        $tenderWorkColumnType = $this->getTenderWorkIdType();

        Schema::create('payment_vouchers', function (Blueprint $table) use ($tenderWorkColumnType) {
            $table->id('Pay_Voucher');
            $table->string('Pay_Vou_No', 50)
                  ->nullable()
                  ->unique()
                  ->comment('Payment Voucher Number - Auto-generated');
            $table->date('bill_challan_date');
            $table->unsignedBigInteger('Br_Id')->comment('Branch ID from main_departments');
            $table->unsignedBigInteger('Lg_Id')->comment('Ledger Account ID');
            $table->unsignedBigInteger('SL_Id')->comment('Sub-Ledger/Vendor ID');
            $table->unsignedBigInteger('Tend_Alloc')->nullable()->comment('Tender Allocation ID');

            // Tender Work ID - Match the data type from new_tender_works.tend_work_id
            if (str_contains(strtolower($tenderWorkColumnType), 'varchar') || str_contains(strtolower($tenderWorkColumnType), 'char')) {
                // If it's a string type, extract length
                preg_match('/varchar\((\d+)\)|char\((\d+)\)/i', $tenderWorkColumnType, $matches);
                $length = !empty($matches[1]) ? $matches[1] : (!empty($matches[2]) ? $matches[2] : 50);
                $table->string('Tend_Work_Id', $length)->nullable()->comment('Tender Work ID from new_tender_works');
            } elseif (str_contains(strtolower($tenderWorkColumnType), 'int')) {
                // If it's an integer type
                $table->unsignedBigInteger('Tend_Work_Id')->nullable()->comment('Tender Work ID from new_tender_works');
            } else {
                // Default to string type
                $table->string('Tend_Work_Id', 50)->nullable()->comment('Tender Work ID from new_tender_works');
            }

            // Amount fields
            $table->decimal('basic_amount', 15, 2)->nullable()->comment('Base amount before taxes');
            $table->decimal('gross_amount', 15, 2)->nullable()->comment('Gross amount');
            $table->decimal('total_tax_amount', 15, 2)->default(0)->comment('Total tax amount');
            $table->decimal('net_amount', 15, 2)->nullable()->comment('Net payable amount');
            $table->decimal('bill_amount', 15, 2)->nullable()->comment('Bill amount (compatibility)');

            // Status and reference fields
            $table->enum('payment_status', ['draft', 'pending', 'approved', 'paid', 'cancelled'])
                  ->default('draft')
                  ->comment('Payment status');
            $table->string('bill_no', 100)->nullable()->comment('Supplier Bill Number');
            $table->string('challan_no', 100)->nullable()->comment('Challan Number');
            $table->text('remarks')->nullable()->comment('Remarks/Notes');

            // JSON fields for tax data
            $table->json('tax_parameters')->nullable()->comment('Tax parameters applied');
            $table->json('tax_calculations')->nullable()->comment('Tax calculations breakdown');

            $table->timestamps();
            $table->softDeletes();

            // Indexes for performance
            $table->index('Br_Id');
            $table->index('Lg_Id');
            $table->index('SL_Id');
            $table->index('Tend_Alloc');
            $table->index('Tend_Work_Id');
            $table->index('bill_challan_date');
            $table->index('Pay_Vou_No');
            $table->index('payment_status');
            $table->index('bill_no');
            $table->index('challan_no');
        });

        // Add foreign keys after table creation
        $this->addForeignKeys();
    }

    /**
     * Get the data type of tend_work_id from new_tender_works table
     */
    private function getTenderWorkIdType(): string
    {
        if (!Schema::hasTable('new_tender_works')) {
            return 'varchar(50)'; // Default if table doesn't exist
        }

        try {
            $columnInfo = DB::select("SHOW COLUMNS FROM new_tender_works WHERE Field = 'tend_work_id'");

            if (!empty($columnInfo)) {
                return $columnInfo[0]->Type;
            }
        } catch (\Exception $e) {
            // If error, return default
        }

        return 'varchar(50)';
    }

    /**
     * Add foreign key constraints
     */
    private function addForeignKeys(): void
    {
        // Wait a moment to ensure table is created
        sleep(1);

        // Add Br_Id foreign key
        if (Schema::hasTable('main_departments')) {
            Schema::table('payment_vouchers', function (Blueprint $table) {
                $table->foreign('Br_Id')->references('id')->on('main_departments')->onDelete('cascade');
            });
        }

        // Add Lg_Id foreign key
        if (Schema::hasTable('ledger_master')) {
            Schema::table('payment_vouchers', function (Blueprint $table) {
                $table->foreign('Lg_Id')->references('Lg_Id')->on('ledger_master')->onDelete('cascade');
            });
        }

        // Add SL_Id foreign key
        if (Schema::hasTable('subledg_master')) {
            Schema::table('payment_vouchers', function (Blueprint $table) {
                $table->foreign('SL_Id')->references('SL_Id')->on('subledg_master')->onDelete('cascade');
            });
        }

        // Add Tend_Alloc foreign key
        if (Schema::hasTable('tendor_allocation')) {
            Schema::table('payment_vouchers', function (Blueprint $table) {
                $table->foreign('Tend_Alloc')->references('Tend_Alloc')->on('tendor_allocation')->onDelete('set null');
            });
        }

        // Add Tend_Work_Id foreign key
        if (Schema::hasTable('new_tender_works')) {
            Schema::table('payment_vouchers', function (Blueprint $table) {
                $table->foreign('Tend_Work_Id')
                      ->references('tend_work_id')
                      ->on('new_tender_works')
                      ->onDelete('set null');
            });
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Drop foreign keys first
        Schema::table('payment_vouchers', function (Blueprint $table) {
            // Safely drop foreign keys
            $foreignKeys = [
                'payment_vouchers_br_id_foreign',
                'payment_vouchers_lg_id_foreign',
                'payment_vouchers_sl_id_foreign',
                'payment_vouchers_tend_alloc_foreign',
                'payment_vouchers_tend_work_id_foreign'
            ];

            foreach ($foreignKeys as $foreignKey) {
                try {
                    $table->dropForeign($foreignKey);
                } catch (\Exception $e) {
                    // Ignore if foreign key doesn't exist
                }
            }
        });

        // Drop the table
        Schema::dropIfExists('payment_vouchers');
    }
};
