class GMCDeviceCapture {
    constructor() {
        this.deviceInfo = {};
    }

    async captureDeviceInfo() {
        try {
            // Basic device info
            this.deviceInfo = {
                user_agent: navigator.userAgent,
                platform: navigator.platform,
                language: navigator.language,
                screen_resolution: `${screen.width}x${screen.height}`,
                timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
                cookies_enabled: navigator.cookieEnabled,
                online_status: navigator.onLine,
                hardware_concurrency: navigator.hardwareConcurrency || 'unknown'
            };

            // Try to get more detailed info (may require user permission)
            await this.getNetworkInfo();
            await this.getHardwareInfo();
            await this.getBrowserFingerprint();

            return this.deviceInfo;

        } catch (error) {
            console.error('Device capture error:', error);
            return this.deviceInfo;
        }
    }

    async getNetworkInfo() {
        try {
            // Get IP address via fetch
            const ipResponse = await fetch('https://api.ipify.org?format=json');
            const ipData = await ipResponse.json();
            this.deviceInfo.ip_address = ipData.ip;
        } catch (error) {
            this.deviceInfo.ip_address = 'unknown';
        }
    }

    async getHardwareInfo() {
        try {
            // Note: These require specific permissions/APIs
            // For desktop apps (Electron/Desktop), you can use system APIs

            // This is simplified - in real app, you'd use:
            // - ActiveX for IE (government systems often use)
            // - Java applet
            // - Desktop app integration
            // - WMI queries for Windows

            this.deviceInfo.mac_address = await this.getMACAddress();
            this.deviceInfo.hdd_serial = await this.getHDDSerial();
            this.deviceInfo.processor_id = await this.getProcessorInfo();

        } catch (error) {
            console.warn('Hardware info not available:', error);
        }
    }

    async getMACAddress() {
        // This is a simplified version
        // In real GMC system, you would use ActiveX or Java applet
        try {
            if (typeof window.ActiveXObject !== 'undefined') {
                // Internet Explorer with ActiveX
                const network = new ActiveXObject("WbemScripting.SWbemLocator");
                const service = network.ConnectServer(".");
                const properties = service.ExecQuery("SELECT * FROM Win32_NetworkAdapterConfiguration");
                const e = new Enumerator(properties);
                for (; !e.atEnd(); e.moveNext()) {
                    const p = e.item();
                    if (p.IPEnabled) {
                        return p.MACAddress;
                    }
                }
            }

            // For other browsers, return placeholder or use other methods
            return 'browser-restricted';
        } catch (error) {
            return 'unknown';
        }
    }

    async getHDDSerial() {
        // Simplified - in real app, use WMI for Windows
        try {
            if (typeof window.ActiveXObject !== 'undefined') {
                const wmi = new ActiveXObject("WbemScripting.SWbemLocator");
                const service = wmi.ConnectServer(".");
                const disks = service.ExecQuery("SELECT * FROM Win32_DiskDrive");
                const e = new Enumerator(disks);
                if (!e.atEnd()) {
                    const disk = e.item();
                    return disk.SerialNumber || 'unknown';
                }
            }
            return 'browser-restricted';
        } catch (error) {
            return 'unknown';
        }
    }

    async getProcessorInfo() {
        try {
            if (typeof window.ActiveXObject !== 'undefined') {
                const wmi = new ActiveXObject("WbemScripting.SWbemLocator");
                const service = wmi.ConnectServer(".");
                const processors = service.ExecQuery("SELECT * FROM Win32_Processor");
                const e = new Enumerator(processors);
                if (!e.atEnd()) {
                    const processor = e.item();
                    return processor.ProcessorId || 'unknown';
                }
            }
            return 'browser-restricted';
        } catch (error) {
            return 'unknown';
        }
    }

    async getBrowserFingerprint() {
        try {
            const components = {
                userAgent: navigator.userAgent,
                platform: navigator.platform,
                language: navigator.language,
                screenWidth: screen.width,
                screenHeight: screen.height,
                colorDepth: screen.colorDepth,
                timezoneOffset: new Date().getTimezoneOffset(),
                sessionStorage: !!window.sessionStorage,
                localStorage: !!window.localStorage,
                indexedDB: !!window.indexedDB,
                cpuClass: navigator.cpuClass || 'unknown',
                doNotTrack: navigator.doNotTrack || 'unknown',
                plugins: this.getPluginsString(),
                fonts: await this.getFonts(),
                canvas: await this.getCanvasFingerprint(),
                webgl: await this.getWebGLFingerprint()
            };

            // Create hash
            const str = JSON.stringify(components);
            this.deviceInfo.browser_fingerprint = await this.hashString(str);

        } catch (error) {
            this.deviceInfo.browser_fingerprint = 'error-generating';
        }
    }

    getPluginsString() {
        const plugins = [];
        for (let i = 0; i < navigator.plugins.length; i++) {
            plugins.push(navigator.plugins[i].name);
        }
        return plugins.join(',');
    }

    async getFonts() {
        // Font detection
        const fonts = [
            'Arial', 'Arial Black', 'Comic Sans MS', 'Courier New',
            'Georgia', 'Impact', 'Times New Roman', 'Trebuchet MS',
            'Verdana', 'Webdings', 'Wingdings'
        ];

        const availableFonts = [];
        const canvas = document.createElement('canvas');
        const context = canvas.getContext('2d');

        for (const font of fonts) {
            context.font = '72px monospace';
            const baseline = context.measureText('mmmmmmmmmmlli').width;
            context.font = '72px ' + font + ', monospace';
            const width = context.measureText('mmmmmmmmmmlli').width;
            if (width !== baseline) {
                availableFonts.push(font);
            }
        }

        return availableFonts.join(',');
    }

    async getCanvasFingerprint() {
        const canvas = document.createElement('canvas');
        const ctx = canvas.getContext('2d');

        canvas.width = 200;
        canvas.height = 50;

        // Draw text
        ctx.textBaseline = 'top';
        ctx.font = '14px Arial';
        ctx.fillStyle = '#f60';
        ctx.fillRect(0, 0, 100, 50);

        ctx.fillStyle = '#069';
        ctx.fillText('GMC Gadhinagar', 2, 15);

        // Get data URL
        return canvas.toDataURL();
    }

    async getWebGLFingerprint() {
        try {
            const canvas = document.createElement('canvas');
            const gl = canvas.getContext('webgl') || canvas.getContext('experimental-webgl');

            if (!gl) return 'no-webgl';

            const debugInfo = gl.getExtension('WEBGL_debug_renderer_info');
            if (debugInfo) {
                return {
                    vendor: gl.getParameter(debugInfo.UNMASKED_VENDOR_WEBGL),
                    renderer: gl.getParameter(debugInfo.UNMASKED_RENDERER_WEBGL)
                };
            }

            return 'webgl-no-debug-info';
        } catch (error) {
            return 'webgl-error';
        }
    }

    async hashString(str) {
        // Simple hash function
        let hash = 0;
        for (let i = 0; i < str.length; i++) {
            const char = str.charCodeAt(i);
            hash = ((hash << 5) - hash) + char;
            hash = hash & hash;
        }
        return Math.abs(hash).toString(16);
    }

    async sendToServer(endpoint) {
        try {
            const response = await fetch(endpoint, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: JSON.stringify(this.deviceInfo)
            });

            return await response.json();
        } catch (error) {
            console.error('Error sending device info:', error);
            throw error;
        }
    }
}

// Usage example for registration form
document.addEventListener('DOMContentLoaded', function() {
    const registrationForm = document.getElementById('registrationForm');

    if (registrationForm) {
        const deviceCapture = new GMCDeviceCapture();

        // Capture device info when form loads
        deviceCapture.captureDeviceInfo().then(info => {
            // Store in hidden field
            const deviceDataField = document.createElement('input');
            deviceDataField.type = 'hidden';
            deviceDataField.name = 'device_data';
            deviceDataField.value = JSON.stringify(info);
            registrationForm.appendChild(deviceDataField);
        });

        // Also send to server endpoint for verification
        registrationForm.addEventListener('submit', async function(e) {
            e.preventDefault();

            // Get latest device info
            const deviceInfo = await deviceCapture.captureDeviceInfo();

            // Add device info to form data
            const formData = new FormData(registrationForm);
            formData.append('device_data', JSON.stringify(deviceInfo));

            // Submit form
            try {
                const response = await fetch(registrationForm.action, {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest'
                    }
                });

                const result = await response.json();

                if (result.success) {
                    alert('Registration successful! ' + result.message);
                    if (result.data.requires_device_approval) {
                        alert('Note: Your device needs admin approval before login.');
                    }
                } else {
                    alert('Registration failed: ' + result.message);
                }
            } catch (error) {
                alert('Network error. Please try again.');
            }
        });
    }
});

// Export for module usage
if (typeof module !== 'undefined' && module.exports) {
    module.exports = GMCDeviceCapture;
}
